#if NUNIT
using System;
using System.Data;
using NUnit.Framework;

using gov.va.med.vbecs.DAL.HL7.Parsers;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;

using gov.va.med.vbecs.UnitTests;

namespace gov.va.med.vbecs.DAL.HL7.UnitTests.Parsers
{
	[TestFixture]
    public class Parsers_CprsProcessPreviousLabResults : BaseTest 
	{
		private static readonly string RBC_COMPONENT_ORDER = "2^RED BLOOD CELLS^99VBC^^";
		private const string KC_DIVISION = "589^VA HEARTLAND - WEST, VISN 15";

		private HL7OmgMessage previousResultsMsg;

		[SetUp]
		public void SetUp()
		{
			previousResultsMsg = HL7UnitTestUtil.GetLabPreviousResultsMessage(); 
		}

		[TearDown]
		public void TearDown()
		{}

		[Test]
		public void CprsProcessPreviousLabResults_Pass_1()
		{
			Assert.IsNotNull( new CprsProcessPreviousLabResults( previousResultsMsg ) );
		}

		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void CprsProcessPreviousLabResults_Fail_1()
		{
			CprsProcessPreviousLabResults nullException = new CprsProcessPreviousLabResults( null );
		}

		[Test]
		public void ProcessPreviousResults_Pass_1()
		{
			HL7OrgMessage ackMessage = new CprsProcessPreviousLabResults( previousResultsMsg ).ProcessPreviousResults();
			if( ackMessage.AcknowledgementCode != AckCodes.AA.ToString() )
				Assert.Fail( "Acknowledgement Code is " + ackMessage.AcknowledgementCode + " and should be AA.");

			string[] orderGroupNumber = HL7Utility.ParseGetStringArray( previousResultsMsg.ORC[4], previousResultsMsg.Delimiters[1].ToString() );

			DataRow drPatientOrder = HL7UnitTestUtil.RunSQLGetDataRow( "SELECT * FROM PatientOrder WHERE OrderGroupNumber = '" + HL7Utility.ConvertString( orderGroupNumber[0] ) + "'" );
			if( drPatientOrder.IsNull("PatientOrderGuid") )
				Assert.Fail( "PatientOrderGuid is null." );
			
			DataTable dtPatientTest = HL7UnitTestUtil.RunSQLGetDataTable( "SELECT * FROM PatientTest WHERE PatientOrderGuid = '" + drPatientOrder["PatientOrderGuid"].ToString() + "'" );
			foreach(DataRow dr in dtPatientTest.Rows)
			{
				if( dr.IsNull("PatientTestGuid") )
					Assert.Fail( "PatientTestGuid is null." );

				Assert.AreEqual( dr["TestResult"].ToString(), previousResultsMsg.OBX[5], "TestResult Mismatch" );

			}
		}

		[Test]
		[ExpectedException( typeof( HL7Exception ) )]
		public void ProcessPreviousResults_Fail_1()
		{
			int orderNum = HL7UnitTestUtil.GetRandomNumber(99999);
			int groupNum = orderNum - 1;
			string messageString = HL7UnitTestUtil.GetCprsMessage( "NW", orderNum.ToString(), groupNum.ToString(), RBC_COMPONENT_ORDER, KC_DIVISION, HL7UnitTestUtil.GetRandomNumber(999999).ToString() );
			HL7OmgMessage omgMessage = new HL7OmgMessage( messageString );

			HL7OrgMessage ackMessage = new CprsProcessPreviousLabResults( omgMessage ).ProcessPreviousResults();
		}

		[Test]
		public void ProcessPreviousResults_Pass_2()
		{
			previousResultsMsg.ORC[4] = HL7UnitTestUtil.GetRandomNumber(99999999) + "^OR";

			HL7OrgMessage ackMessage = new CprsProcessPreviousLabResults( previousResultsMsg ).ProcessPreviousResults();

			Assert.AreEqual("AR", ackMessage.AcknowledgementCode, "Ack Check" );

		}

	}
}
#endif